<?php

namespace App\Http\Controllers;

use App\Models\Entry;
use App\Models\TemporaryQrCode;
use Illuminate\Http\Request;
use App\Models\Request as RequestModel;

class EntryController extends Controller
{

    public function store(Request $request)
    {
        $user = auth()->user();

        $request->validate([
            'place_id' => 'required|exists:places,id',
            'qr_code' => 'required|string',
            'entry_type' => 'required|in:entry,exit',
        ]);

        // چک درخواست تایید شده
        $hasApproved = RequestModel::where('user_id', $user->id)
            ->where('place_id', $request->place_id)
            ->where('status', 'approved')
            ->exists();

        if (!$hasApproved) {
            return response()->json([
                'message' => 'شما اجازه ورود به این مکان را ندارید.',
                'status' => 'forbidden'
            ], 403);
        }

        // چک QR
        $qr = TemporaryQrCode::where('code', $request->qr_code)
            ->where('user_id', $user->id)
            ->where('place_id', $request->place_id)
            ->where('is_used', false)
            ->where('expires_at', '>', now())
            ->first();

        if (!$qr) {
            return response()->json([
                'message' => 'QR Code نامعتبر یا منقضی شده.',
                'status' => 'invalid_or_expired'
            ], 400);
        }

        // چک تکراری نبودن ورود/خروج در همان روز
        $exists = Entry::where('user_id', $user->id)
            ->where('place_id', $request->place_id)
            ->where('entry_type', $request->entry_type)
            ->whereDate('created_at', now()->toDateString())
            ->exists();

        if ($exists) {
            $typeLabel = $request->entry_type === 'entry' ? 'ورود' : 'خروج';
            return response()->json([
                'message' => "شما امروز قبلاً $typeLabel ثبت کرده‌اید و نمی‌توانید مجدداً ثبت کنید.",
                'status' => 'already_exists'
            ], 400);
        }

        // ثبت ورود یا خروج
        $entry = new Entry();
        $entry->user_id = $user->id;
        $entry->place_id = $request->place_id;
        $entry->qr_code_id = $qr->code;
        $entry->entry_type = $request->entry_type;
        $entry->entry_time = now();
        $entry->save();

        // غیرفعال کردن QR
        $qr->is_used = true;
        $qr->save();

        return response()->json([
            'message' => 'حضور با موفقیت ثبت شد.',
            'status' => 'success'
        ], 200);
    }


}
