<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Operator;
use App\Models\Place;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class OperatorController extends Controller
{
    // لیست اپراتورها
    public function index()
    {
        $operators = Operator::with('place:id,name') // فرض بر اینکه رابطه تعریف شده
        ->select('id', 'username', 'role', 'place_id', 'is_active', 'created_at')
            ->get();

        return response()->json($operators);
    }

    // افزودن اپراتور جدید
    public function store(Request $request)
    {
        $request->validate([
            'username' => 'required|string|max:255|unique:operators,username',
            'password' => 'required|string|min:6',
            'role'     => ['required', Rule::in(['system_operator', 'facility_operator'])],
            'place_id' => 'nullable|exists:places,id',
            'is_active' => 'boolean'
        ]);

        $operator = Operator::create([
            'username' => $request->username,
            'password_hash' => Hash::make($request->password),
            'role' => $request->role,
            'place_id' => $request->place_id,
            'is_active' => $request->is_active ?? 1
        ]);

        return response()->json(['message' => 'اپراتور با موفقیت اضافه شد.', 'operator' => $operator], 201);
    }

    // ویرایش اپراتور
    public function update(Request $request, $id)
    {
        $operator = Operator::findOrFail($id);

        $request->validate([
            'username' => 'required|string|max:255|unique:operators,username,' . $id,
            'role'     => ['required', Rule::in(['system_operator', 'facility_operator'])],
            'place_id' => 'nullable|exists:places,id',
            'is_active' => 'boolean'
        ]);

        $operator->update([
            'username' => $request->username,
            'role' => $request->role,
            'place_id' => $request->place_id,
            'is_active' => $request->is_active ?? $operator->is_active
        ]);

        return response()->json(['message' => 'اپراتور با موفقیت ویرایش شد.']);
    }

    // فعال/غیرفعال‌سازی اپراتور
    public function toggleStatus($id)
    {
        $operator = Operator::findOrFail($id);
        $operator->is_active = !$operator->is_active;
        $operator->save();

        return response()->json(['message' => 'وضعیت اپراتور تغییر کرد.']);
    }

    // حذف اپراتور
    public function destroy($id)
    {
        $operator = Operator::findOrFail($id);
        $operator->delete();

        return response()->json(['message' => 'اپراتور با موفقیت حذف شد.']);
    }

    // تغییر پسورد
    public function changePassword(Request $request, $id)
    {
        $operator = Operator::findOrFail($id);

        $request->validate([
            'new_password' => 'required|string|min:6|confirmed'
        ]);

        $operator->password_hash = Hash::make($request->new_password);
        $operator->save();

        return response()->json(['message' => 'پسورد اپراتور با موفقیت تغییر کرد.']);
    }
}
