<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Request as UserRequest;
use Illuminate\Support\Facades\Auth;

class OperatorRequestController extends Controller
{
    /**
     * نمایش لیست درخواست‌های در انتظار تأیید (فقط برای مدیر سیستم)
     */
    public function listRequests()
    {
        // دریافت اپراتور لاگین شده
        $operator = Auth::guard('operator_api')->user();

        // فقط مدیر سیستم (`system_operator`) می‌تواند درخواست‌ها را ببیند
        if ($operator->role !== 'system_operator') {
            return response()->json(['message' => 'شما اجازه مشاهده این درخواست‌ها را ندارید.'], 403);
        }

        // دریافت تمام درخواست‌های در انتظار تأیید
        $requests = UserRequest::with(['user:id,full_name,personnel_code', 'place:id,name'])
            ->get();

        return response()->json($requests, 200);
    }

    /**
     * تأیید یک درخواست ورود (فقط مدیر سیستم)
     */
    public function approveRequest(Request $request)
    {
        $validated = $request->validate([
            'request_id' => 'required|exists:requests,id',
        ]);

        $operator = Auth::guard('operator_api')->user();

        // فقط مدیر سیستم (`system_operator`) می‌تواند درخواست را تأیید کند
        if ($operator->role !== 'system_operator') {
            return response()->json(['message' => 'شما اجازه تأیید این درخواست را ندارید.'], 403);
        }

        $userRequest = UserRequest::findOrFail($validated['request_id']);
        $userRequest->update([
            'status' => 'approved',
            'approved_at' => now(),
            'operator_id' => $operator->id,
        ]);

        return response()->json(['message' => 'درخواست تأیید شد.'], 200);
    }

    /**
     * رد یک درخواست ورود (فقط مدیر سیستم)
     */
    public function rejectRequest(Request $request)
    {
        $validated = $request->validate([
            'request_id' => 'required|exists:requests,id',
        ]);

        $operator = Auth::guard('operator_api')->user();

        // فقط مدیر سیستم (`system_operator`) می‌تواند درخواست را رد کند
        if ($operator->role !== 'system_operator') {
            return response()->json(['message' => 'شما اجازه رد این درخواست را ندارید.'], 403);
        }

        $userRequest = UserRequest::findOrFail($validated['request_id']);
        $userRequest->update([
            'status' => 'rejected',
            'approved_at' => now(),
            'operator_id' => $operator->id,
        ]);

        return response()->json(['message' => 'درخواست رد شد.'], 200);
    }
}
